<?php
require_once '../config.php';
require_once '../functions.php';
requireLogin();

$analysisData = [];
$chartData = [];

if (checkDataFile()) {
    $data = readCSVData();
    $engineerData = [];
    
    // Process all rows
    foreach ($data as $row) {
        $engName = $row[16] ?? 'Unknown';
        $workTime = floatval($row[24] ?? 0);
        $travelTime = floatval($row[25] ?? 0);
        
        if (!isset($engineerData[$engName])) {
            $engineerData[$engName] = [
                'total_work' => 0,
                'total_travel' => 0
            ];
        }
        
        $engineerData[$engName]['total_work'] += $workTime;
        $engineerData[$engName]['total_travel'] += $travelTime;
    }
    
    // Calculate final metrics
    foreach ($engineerData as $name => $data) {
        $totalTime = $data['total_work'] + $data['total_travel'];
        $travelShare = $totalTime > 0 ? $data['total_travel'] / $totalTime : 0;
        
        $analysisData[] = [
            'engineer' => $name,
            'work_hours' => $data['total_work'],
            'travel_hours' => $data['total_travel'],
            'total_hours' => $totalTime,
            'travel_share' => $travelShare
        ];
    }
    
    // Sort by travel share descending (highest travel share first)
    usort($analysisData, function($a, $b) {
        return $b['travel_share'] <=> $a['travel_share'];
    });
    
    // Prepare chart data (top 10 by travel share)
    $top10 = array_slice($analysisData, 0, 10);
    foreach ($top10 as $item) {
        $chartData['engineers'][] = $item['engineer'];
        $chartData['work'][] = round($item['work_hours'], 2);
        $chartData['travel'][] = round($item['travel_hours'], 2);
        $chartData['travel_share'][] = round($item['travel_share'] * 100, 1);
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Travel Efficiency - <?php echo APP_NAME; ?></title>
    <link href="https://fonts.googleapis.com/css2?family=Manrope:wght@400;500;600;700&display=swap" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Manrope', sans-serif; background: #f8fafc; color: #1e293b; }
        .container { display: flex; min-height: 100vh; }
        
        .sidebar { width: 260px; background: white; border-right: 1px solid #e2e8f0; padding: 24px 0; position: fixed; height: 100vh; overflow-y: auto; }
        .sidebar-header { padding: 0 24px 24px; border-bottom: 1px solid #e2e8f0; display: flex; align-items: center; gap: 12px; }
        .logo { height: 32px; }
        .app-name { font-size: 20px; font-weight: 700; color: #0066b2; }
        .menu { list-style: none; padding: 16px 0; }
        .menu li { margin-bottom: 4px; }
        .menu a { display: flex; align-items: center; gap: 12px; padding: 12px 24px; color: #64748b; text-decoration: none; font-size: 15px; font-weight: 500; transition: all 0.2s; }
        .menu a:hover, .menu a.active { background: #f1f5f9; color: #0066b2; }
        .menu-icon { width: 20px; height: 20px; display: flex; align-items: center; justify-content: center; }
        
        .main-content { flex: 1; margin-left: 260px; padding: 32px; }
        .topbar { background: white; border-radius: 12px; padding: 16px 24px; margin-bottom: 32px; display: flex; justify-content: space-between; align-items: center; box-shadow: 0 1px 3px rgba(0,0,0,0.05); }
        .page-title { font-size: 28px; font-weight: 700; color: #0f172a; display: flex; align-items: center; gap: 12px; }
        .help-badge { width: 28px; height: 28px; background: #dbeafe; color: #2563eb; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 14px; font-weight: 700; cursor: help; }
        .user-info { display: flex; align-items: center; gap: 12px; }
        .user-avatar { width: 40px; height: 40px; background: #0066b2; color: white; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-weight: 600; font-size: 14px; }
        .user-details { text-align: right; }
        .user-name { font-weight: 600; font-size: 14px; color: #0f172a; }
        .user-role { font-size: 12px; color: #64748b; }
        
        .info-box { background: #dbeafe; border-left: 4px solid #2563eb; padding: 20px; border-radius: 8px; margin-bottom: 32px; }
        .info-box-title { font-weight: 700; color: #0f172a; margin-bottom: 8px; font-size: 16px; }
        .info-box-content { color: #334155; font-size: 14px; line-height: 1.6; }
        .info-box ul { margin-left: 20px; margin-top: 8px; }
        .info-box li { margin-bottom: 4px; }
        
        .chart-section { background: white; border-radius: 12px; padding: 24px; box-shadow: 0 1px 3px rgba(0,0,0,0.05); margin-bottom: 32px; }
        .chart-title { font-size: 20px; font-weight: 700; color: #0f172a; margin-bottom: 20px; }
        .chart-container { position: relative; height: 400px; }
        
        .table-section { background: white; border-radius: 12px; padding: 24px; box-shadow: 0 1px 3px rgba(0,0,0,0.05); overflow-x: auto; }
        table { width: 100%; border-collapse: collapse; }
        thead { background: #f8fafc; position: sticky; top: 0; }
        th { text-align: left; padding: 12px 16px; font-size: 12px; font-weight: 600; color: #64748b; text-transform: uppercase; letter-spacing: 0.5px; white-space: nowrap; }
        td { padding: 12px 16px; border-bottom: 1px solid #f1f5f9; font-size: 14px; color: #334155; white-space: nowrap; }
        tbody tr:hover { background: #f8fafc; }
        .badge { display: inline-block; padding: 4px 10px; border-radius: 6px; font-size: 12px; font-weight: 600; }
        .badge-success { background: #d1fae5; color: #10b981; }
        .badge-warning { background: #fef3c7; color: #f59e0b; }
        .badge-danger { background: #fee2e2; color: #dc2626; }
        .text-right { text-align: right; }
    </style>
</head>
<body>
    <div class="container">
        <aside class="sidebar">
            <div class="sidebar-header">
                <img src="<?php echo LOGO_URL; ?>" alt="Logo" class="logo">
                <span class="app-name"><?php echo APP_NAME; ?></span>
            </div>
            <ul class="menu">
                <li><a href="../index.php"><span class="menu-icon">📊</span> Dashboard</a></li>
                <li><a href="engineer_performance.php"><span class="menu-icon">👤</span> Engineer Performance</a></li>
                <li><a href="contract_view.php"><span class="menu-icon">📄</span> Contract Analysis</a></li>
                <li><a href="travel_efficiency.php" class="active"><span class="menu-icon">🚗</span> Travel Efficiency</a></li>
                <li><a href="documentation.php"><span class="menu-icon">📝</span> Documentation</a></li>
                <li><a href="../logout.php"><span class="menu-icon">🚪</span> Logout</a></li>
            </ul>
        </aside>
        
        <main class="main-content">
            <div class="topbar">
                <h1 class="page-title">
                    🚗 Travel vs Work Efficiency
                    <span class="help-badge" title="View analysis methodology">?</span>
                </h1>
                <div class="user-info">
                    <div class="user-avatar"><?php echo strtoupper(substr($_SESSION['full_name'], 0, 2)); ?></div>
                    <div class="user-details">
                        <div class="user-name"><?php echo $_SESSION['full_name']; ?></div>
                        <div class="user-role"><?php echo ucfirst($_SESSION['role']); ?></div>
                    </div>
                </div>
            </div>
            
            <div class="info-box">
                <div class="info-box-title">📊 Analysis Methodology</div>
                <div class="info-box-content">
                    This analysis provides engineer-level travel vs work efficiency metrics. For each engineer, the following is calculated:
                    <ul>
                        <li><strong>Work Hours:</strong> Total time spent on actual work tasks (WORK_TIME column)</li>
                        <li><strong>Travel Hours:</strong> Total time spent traveling to/from job sites (TRAVELLING_TIME column)</li>
                        <li><strong>Total Hours:</strong> Combined work and travel time</li>
                        <li><strong>Travel Share:</strong> Percentage of total time spent traveling, calculated as: Travel Hours / (Work Hours + Travel Hours)</li>
                    </ul>
                    Engineers with high travel share percentages may benefit from route optimization or job scheduling improvements.
                </div>
            </div>
            
            <?php if (!empty($chartData)): ?>
            <div class="chart-section">
                <div class="chart-title">Top 10 Engineers by Travel Share (%)</div>
                <div class="chart-container">
                    <canvas id="travelShareChart"></canvas>
                </div>
            </div>
            
            <div class="chart-section">
                <div class="chart-title">Work vs Travel Hours Comparison</div>
                <div class="chart-container">
                    <canvas id="comparisonChart"></canvas>
                </div>
            </div>
            <?php endif; ?>
            
            <div class="table-section">
                <div class="chart-title">Complete Travel Efficiency Data</div>
                <table>
                    <thead>
                        <tr>
                            <th>ENGINEER</th>
                            <th class="text-right">WORK HOURS</th>
                            <th class="text-right">TRAVEL HOURS</th>
                            <th class="text-right">TOTAL HOURS</th>
                            <th class="text-right">TRAVEL SHARE</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($analysisData as $row): ?>
                        <tr>
                            <td><strong><?php echo htmlspecialchars($row['engineer']); ?></strong></td>
                            <td class="text-right"><?php echo formatNumber($row['work_hours'], 1); ?></td>
                            <td class="text-right"><?php echo formatNumber($row['travel_hours'], 1); ?></td>
                            <td class="text-right"><?php echo formatNumber($row['total_hours'], 1); ?></td>
                            <td class="text-right">
                                <?php if ($row['travel_share'] > 0.4): ?>
                                <span class="badge badge-danger"><?php echo formatPercent($row['travel_share']); ?></span>
                                <?php elseif ($row['travel_share'] > 0.25): ?>
                                <span class="badge badge-warning"><?php echo formatPercent($row['travel_share']); ?></span>
                                <?php else: ?>
                                <span class="badge badge-success"><?php echo formatPercent($row['travel_share']); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </main>
    </div>
    
    <?php if (!empty($chartData)): ?>
    <script>
        // Travel Share Chart
        const travelShareCtx = document.getElementById('travelShareChart').getContext('2d');
        new Chart(travelShareCtx, {
            type: 'bar',
            data: {
                labels: <?php echo json_encode($chartData['engineers']); ?>,
                datasets: [{
                    label: 'Travel Share (%)',
                    data: <?php echo json_encode($chartData['travel_share']); ?>,
                    backgroundColor: 'rgba(245, 158, 11, 0.8)',
                    borderColor: 'rgba(245, 158, 11, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return context.parsed.y.toFixed(1) + '%';
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        max: 100,
                        title: { display: true, text: 'Travel Share (%)' }
                    }
                }
            }
        });
        
        // Work vs Travel Comparison Chart
        const comparisonCtx = document.getElementById('comparisonChart').getContext('2d');
        new Chart(comparisonCtx, {
            type: 'bar',
            data: {
                labels: <?php echo json_encode($chartData['engineers']); ?>,
                datasets: [
                    {
                        label: 'Work Hours',
                        data: <?php echo json_encode($chartData['work']); ?>,
                        backgroundColor: 'rgba(16, 185, 129, 0.8)',
                        borderColor: 'rgba(16, 185, 129, 1)',
                        borderWidth: 1
                    },
                    {
                        label: 'Travel Hours',
                        data: <?php echo json_encode($chartData['travel']); ?>,
                        backgroundColor: 'rgba(220, 38, 38, 0.8)',
                        borderColor: 'rgba(220, 38, 38, 1)',
                        borderWidth: 1
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { position: 'top' },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return context.dataset.label + ': ' + context.parsed.y.toFixed(1) + ' hours';
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        stacked: false,
                        title: { display: true, text: 'Hours' }
                    }
                }
            }
        });
    </script>
    <?php endif; ?>
</body>
</html>
